const express = require("express");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { PrismaClient } = require("@prisma/client");
const crypto = require("crypto");
const { sendEmail } = require("../utils/mailer");
const {
  verifyEmailTemplate,
  resetPasswordTemplate,
} = require("../utils/emailTemplates");

const router = express.Router();
const prisma = new PrismaClient();

const JWT_SECRET = process.env.JWT_SECRET || "dev_secret";
const siteUrl = process.env.SITE_URL || "http://localhost:3000";

// Register
router.post("/register", async (req, res) => {
  const { email, password, name, role } = req.body;
  if (!email || !password || !name || !role) {
    return res.status(400).json({ error: "All fields are required." });
  }
  const existing = await prisma.user.findUnique({ where: { email } });
  if (existing) return res.status(400).json({ error: "Email already exists." });

  const passwordHash = await bcrypt.hash(password, 10);
  const user = await prisma.user.create({
    data: {
      name,
      email,
      passwordHash,
      role,
      email_verified: false,
    },
  });

  // Create verification token
  const token = crypto.randomBytes(32).toString("hex");
  const expiresAt = new Date(Date.now() + 1000 * 60 * 60 * 24); // 24h
  await prisma.emailVerification.create({
    data: { userId: user.id, token, expiresAt, used: false },
  });

  // Send verification email
  await sendEmail(
    user.email,
    "Verify your email",
    `Please verify your email: ${siteUrl}/verify-email?token=${token}`,
    verifyEmailTemplate(`${siteUrl}/verify-email?token=${token}`)
  );
  console.log("Email sent successfully");

  res.json({
    success: true,
    message: "Registration successful. Please verify your email.",
  });
});

// Login
router.post("/login", async (req, res) => {
  const { email, password } = req.body;
  const user = await prisma.user.findUnique({ where: { email } });
  if (!user) return res.status(400).json({ error: "Invalid credentials." });
  const valid = await bcrypt.compare(password, user.passwordHash);
  if (!valid) return res.status(400).json({ error: "Invalid credentials." });
  if (!user.email_verified) {
    return res
      .status(403)
      .json({ success: false, message: "Email not verified" });
  }

  const token = jwt.sign({ userId: user.id, role: user.role }, JWT_SECRET, {
    expiresIn: "7d",
  });
  res.json({
    success: true,
    token,
    user: { id: user.id, email: user.email, role: user.role, name: user.name },
  });
});

// Verify Email
router.post("/verify-email", async (req, res) => {
  const { token } = req.body;
  const record = await prisma.emailVerification.findUnique({
    where: { token },
  });
  if (!record || record.used || record.expiresAt < new Date()) {
    return res.status(400).json({ error: "Invalid or expired token." });
  }
  await prisma.user.update({
    where: { id: record.userId },
    data: { email_verified: true },
  });
  await prisma.emailVerification.update({
    where: { token },
    data: { used: true },
  });
  res.json({ success: true, message: "Email verified." });
});

// Resend Verification
router.post("/resend-verification", async (req, res) => {
  const { email } = req.body;
  const user = await prisma.user.findUnique({ where: { email } });
  if (!user) return res.status(400).json({ error: "User not found." });
  if (user.email_verified)
    return res.status(400).json({ error: "Already verified." });

  const token = crypto.randomBytes(32).toString("hex");
  const expiresAt = new Date(Date.now() + 1000 * 60 * 60 * 24);
  await prisma.emailVerification.create({
    data: { userId: user.id, token, expiresAt, used: false },
  });

  await sendEmail(
    user.email,
    "Verify your email",
    `Please verify your email: ${siteUrl}/verify-email?token=${token}`,
    verifyEmailTemplate(`${siteUrl}/verify-email?token=${token}`)
  );

  res.json({ success: true, message: "Verification email resent." });
});

// Forgot Password
router.post("/forgot-password", async (req, res) => {
  const { email } = req.body;
  const user = await prisma.user.findUnique({ where: { email } });
  if (!user) return res.status(400).json({ error: "User not found." });

  const token = crypto.randomBytes(32).toString("hex");
  const expiresAt = new Date(Date.now() + 1000 * 60 * 60); // 1 hour
  await prisma.passwordReset.create({
    data: { userId: user.id, token, expiresAt, used: false },
  });

  await sendEmail(
    user.email,
    "Reset your password",
    `Reset your password: ${siteUrl}/reset-password?token=${token}`,
    resetPasswordTemplate(`${siteUrl}/reset-password?token=${token}`)
  );

  res.json({ success: true, message: "Password reset email sent." });
});

// Reset Password
router.post("/reset-password", async (req, res) => {
  const { token, password } = req.body;
  const record = await prisma.passwordReset.findUnique({ where: { token } });
  if (!record || record.used || record.expiresAt < new Date()) {
    return res.status(400).json({ error: "Invalid or expired token." });
  }
  const passwordHash = await bcrypt.hash(password, 10);
  await prisma.user.update({
    where: { id: record.userId },
    data: { passwordHash },
  });
  await prisma.passwordReset.update({ where: { token }, data: { used: true } });
  res.json({ success: true, message: "Password reset successful." });
});

// Add this endpoint
router.get("/me", async (req, res) => {
  const token = req.headers.authorization?.split(" ")[1];
  if (!token) return res.status(401).json({ error: "No token provided" });

  try {
    const decoded = jwt.verify(token, JWT_SECRET);
    const user = await prisma.user.findUnique({
      where: { id: decoded.userId },
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        email_verified: true,
      },
    });
    if (!user) throw new Error("User not found");
    res.json({ user });
  } catch (error) {
    res.status(401).json({ error: "Invalid token" });
  }
});

// Add this route after your existing routes
/* router.post("/create-admin", async (req, res) => {
  const { email, password, secretKey } = req.body;

  // Verify secret key - make this complex and temporary
  if (secretKey !== process.env.ADMIN_SECRET_KEY) {
    return res.status(403).json({ error: "Invalid secret key" });
  }

  try {
    const existing = await prisma.user.findUnique({ where: { email } });
    if (existing) {
      return res.status(400).json({ error: "Email already exists" });
    }

    const passwordHash = await bcrypt.hash(password, 10);
    const admin = await prisma.user.create({
      data: {
        email,
        passwordHash,
        name: "Admin User",
        role: "admin",
        email_verified: true, // Admin is pre-verified
      },
    });

    res.json({
      success: true,
      message: "Admin created successfully",
      admin: { id: admin.id, email: admin.email },
    });
  } catch (error) {
    res.status(500).json({ error: "Failed to create admin" });
  }
}); */

module.exports = router;
